/**
 * \file            field_debug.h
 * \brief           This file contains field_debug related definitions
 *                  It has been simplified to simply print to the console
 *                  Ported from the AIP/DVPHD
 *
 * \author          Adolfo Velasco
 *                  Yun Mao

 * \date            06/16/08
 * \warning
 * \note
 * \review
 **/

#include "nutypedefs.h"
#include "field_debug.h"
#include "console.h"
#include "string_utils.h"               // for stricmp
#include <string.h>
#include <stdlib.h>

//Global field debug state flag
UINT32 g_lDmFieldDebugState[DM_FIELD_DEBUG_ARRAY_SIZE] = {0, 0};

//Bugzilla fix
//Protect against lazy people that use "FDEBUG ON" by not automatically enabling debugs that can mess up system timing
const UINT32 g_lDmFieldDebugMaskPeopleReilly[DM_FIELD_DEBUG_ARRAY_SIZE] =
{
    0x400000C0,     //DM_FIELD_CRESNET_RX_IDX, DM_FIELD_CRESNET_TX_IDX, DM_FIELD_OW_POLL_DEBUG_IDX
    0x00000000
};

const char * const fieldDebugNames[MAX_DM_FIELD_DEBUG_NUM - 1] =
{
  "HDMI RX",                           //1   DM_FIELD_HDMI_RX_DEBUG_IDX = 1,
  "HDMI TX",                           //2   DM_FIELD_HDMI_TX_DEBUG_IDX,
  "ANALOG RX",                         //3   DM_FIELD_ANALOG_RX_DEBUG_IDX,
  "HDCP MAX DEVICE TEST",              //4   DM_FIELD_HDCP_DEV_SIZE_TEST_DEBUG_IDX,
  "CP VERIFICATION",                   //5   DM_FIELD_HDCP_CP_VERIFICATION_IDX,
  "EDID",                              //6   DM_FIELD_DEBUG_EDID_IDX,
  "CRESNET RX",                        //7   DM_FIELD_CRESNET_RX_IDX,
  "CRESNET TX",                        //8   DM_FIELD_CRESNET_TX_IDX,
  "HDMI RX AUDIO",                     //9   DM_FIELD_HDMI_RX_AUDIO_DEBUG_IDX,
  "DSP",                               //10  DM_FIELD_DSP_DEBUG_IDX,
  "HDMI FIBER",                        //11  DM_FIELD_HDMI_FIBER_DEBUG_IDX,
  "USB",                               //12  DM_FIELD_USB_DEBUG_IDX,
  "HDCP RX",                           //13  DM_FIELD_HDCP_RX_DEBUG_IDX,
  "HDCP TX",                           //14  DM_FIELD_HDCP_TX_DEBUG_IDX,
  "HDMI TX AUDIO",                     //15  DM_FIELD_HDMI_TX_AUDIO_DEBUG_IDX,
  "STREAM",                            //16  DM_FIELD_STREAM_DEBUG_IDX,
  "DM RX",                             //17  DM_FIELD_DM_RX_DEBUG_IDX,
  "DM TX",                             //18  DM_FIELD_DM_TX_DEBUG_IDX,
  "SERIAL PORT",                       //19  DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX,
  "DM MASTER",                         //20  DM_FIELD_DM_MASTER_DEBUG_IDX,
  "DM SLAVE",                          //21  DM_FIELD_DM_SLAVE_DEBUG_IDX,
  "ETHERNET",                          //22  DM_FIELD_ETHERNET_DEBUG_IDX,
  "DMNET CEC",                         //23  DM_FIELD_DMNET_CEC_DEBUG_IDX,
  "SDI RX",                            //24  DM_FIELD_SDI_RX_DEBUG_IDX,
  "SCALER",                            //25  DM_FIELD_SCALER_DEBUG_IDX,
  "ANALOG TX",                         //26  DM_FIELD_ANALOG_TX_DEBUG_IDX,
  "OW RX",                             //27  DM_FIELD_OW_RX_DEBUG_IDX,
  "OW TX",                             //28  DM_FIELD_OW_TX_DEBUG_IDX,
  "VS RX",                             //29  DM_FIELD_VS_RX_DEBUG_IDX,
  "VS TX",                             //30  DM_FIELD_VS_TX_DEBUG_IDX,
  "OW POLL",                           //31  DM_FIELD_OW_POLL_DEBUG_IDX,
  "OW EDID",                           //32  DM_FIELD_OW_EDID_DEBUG_IDX,
  "OW HDCP",                           //33  DM_FIELD_OW_HDCP_DEBUG_IDX,
  "OVERLAY HANDLER",                   //34  DM_FIELD_OVERLAY_HANDLER_IDX,
  "LVDS RX",                           //35  DM_FIELD_LVDS_RX_IDX,
  "HDMI CEC",                          //36  DM_FIELD_CEC_IDX,
  "AUDIO DAC",                         //37  DM_FIELD_AUDIO_DAC_IDX,
  "FPGA",                              //38  DM_FIELD_FPGA_IDX,
  "ENCAPSULATION",                     //39  DM_FIELD_ENCAPSULATION_IDX,
  "SWITCHER",                          //40  DM_FIELD_SWITCHER_IDX,
  "DP RX",                             //41  DM_FIELD_DP_RX_IDX,
  "TOUCH",                             //42  DM_FIELD_TOUCH_DEBUG_IDX,
  "CALTOUCH",                          //43  DM_FIELD_TOUCH_CALIBRATE_DEBUG_IDX,
  "IR TX",                             //44  DM_FIELD_IR_TX_IDX,
  "DM CONTROLLER",                     //45  DM_FIELD_DM_CONTROLLER_IDX,
  "SYSTEM STATS",                      //46  DM_FIELD_SYSTEM_STATS_IDX,
  "EXT TOOL EDID",                     //47  DM_FIELD_DEBUG_EXT_TOOL_EDID_IDX,
  "USB I",                             //48  DM_FIELD_USB_ICRON_DEBUG_IDX,
  "VIDEO MSG HANDLER",                 //49  DM_FIELD_VIDEO_MESSAGE_HANDLER_DEBUG_IDX,
  "FRONT PANEL",                       //50  DM_FIELD_FRONT_PANEL_DEBUG_IDX,
  "CRESNET MASTER",                    //51  DM_FIELD_CNET_MASTER_DEBUG_IDX,
  "LCDMAN",                            //52  DM_FIELD_LCD_MANAGER_DEBUG_IDX,
  "INTERAPP",                          //53  DM_FIELD_INTERAPP_DEBUG_IDX,
  "TEST GEN",                          //54  DM_FIELD_TESTGEN_DEBUG_IDX,
  "AMP AUDIO",                         //55  DM_FIELD_AMP_AUDIO_DEBUG_IDX,
  "CHANNEL MGR",                       //56  DM_FIELD_CHANNELMGR_DEBUG_IDX,
  "AVM MGR",                           //57  DM_FIELD_4K_AVM_DEBUG_IDX ,
  "AVM CNET MGR",                      //58  DM_FIELD_4K_AVM_CNET_DEBUG_IDX,
  "I2C",                               //59  DM_FIELD_I2C_DEBUG_IDX,
};


/**
* \author         Adolfo Velasco
*                 Yun Mao
* \brief          Prints field debug help
* \date           06/20/08
* \param
* \return         none
* \retval         none
*/
void WriteFieldDebugHelp(void)
{
    DmConsolePrintf("To use field debug: FDEBUG [INDEX] [ON|OFF]\r");
    for (int i = 0; i < MAX_DM_FIELD_DEBUG_NUM - 1; i++)
        DmConsolePrintf("%s\r", fieldDebugNames[i]);
    DmConsolePrintf("\r\nTo disable all field debugs: FDEBUG OFF\r");
    DmConsolePrintf("To enable all field debugs: FDEBUG ON\r");
}

/**
* \author         Adolfo Velasco
*                 Yun Mao
* \brief          Prints the current field debug state
* \date           06/20/08
* \param
* \return         none
* \retval         none
*/
void WriteFieldDebugMsg(void)
{
    for (int i = 0; i < MAX_DM_FIELD_DEBUG_NUM - 1; i++)
    {
        DmConsolePrintf("%02d %s %s\r", i+1, ( (IsDmFieldDebugIndexActive(i+1) ? "ON " : "OFF")), fieldDebugNames[i] );
    }
}

/**
* \author         Adolfo Velasco
*                 Yun Mao
* \brief          Parses the field debug console command
* \date           06/20/08
* \param
* \return         none
* \retval         none
*/
INT32 DmFieldDebugCmd(UINT32 argc, char *cmd)
{
    UINT8 i;
    INT8 bIndex;
    INT8 bEnable;
    char *pText;

    //Check for no arguments and print the current field debug state
    if ( !cmd || !(*cmd))
    {
        WriteFieldDebugMsg();
        return 0;
    }
    //Check for the help string
    else if ( (*cmd == '?') )
    {
        //Print the help and exit
        WriteFieldDebugHelp();
        return 0;
    }

    //Check the first argument
    pText = strtok(cmd," ");
    if (!pText)
    {
        //Print the help and exit
        WriteFieldDebugHelp();
        return 0;
    }

    //First argument exists so process it

    //Disable all field debug states
    if ( !stricmp(pText,"OFF") )
    {
        //Clear all bits
        for ( i = 0; i < DM_FIELD_DEBUG_ARRAY_SIZE; i++ )
        {
            g_lDmFieldDebugState[i] = 0;
        }
        DmConsolePrintf("ALL FIELD DEBUG DISABLED\r");
    }
    //Enable all field debug states
    else if ( !stricmp(pText,"ON") )
    {
        // set most bits
        for ( i = 0; i < DM_FIELD_DEBUG_ARRAY_SIZE; i++ )
        {
            g_lDmFieldDebugState[i] |= (0xFFFFFFFF & ~g_lDmFieldDebugMaskPeopleReilly[i]);
        }
        DmConsolePrintf("ALL FIELD DEBUG ENABLED\r");
    }
    else
    {
        //Get the field debug index and verify it
        bIndex = atoi(pText);
        if ( !IsValidDmFieldIndex(bIndex) )
        {
            //Print the help and exit
            WriteFieldDebugHelp();
            return 0;
        }

        //Look for the second argument (ON/OFF)
        pText = strtok(NULL, " ");
        if (pText)
        {
            //Second argument exists but assume it is invalid
            bEnable = -1;

            //User wants to turn it off
            if ( !stricmp(pText,"OFF") )
            {
                bEnable = 0;
            }
            //User wants to turn it on
            else if ( !stricmp(pText,"ON") )
            {
                bEnable = 1;
            }

            //Only process valid 2nd parameters
            if ( bEnable == -1 )
            {
                //Print the help and exit
                WriteFieldDebugHelp();
                return 0;
            }

            //Validate the index
            if ( IsValidDmFieldIndex(bIndex) )
            {
                //Enable/disable accordingly
                if ( bEnable )
                {
                    EnableDmFieldDebugIndex(bIndex);
                }
                else
                {
                    DisableDmFieldDebugIndex(bIndex);
                }

                DmConsolePrintf("FIELD DEBUG %s %sABLED\r", fieldDebugNames[bIndex-1], bEnable ? "EN" : "DIS");
            }
        }
    }

    //Update the field debug parameters
    UpdateFieldDebugParam();
    return 0;
}


